document.addEventListener("DOMContentLoaded", () => {
    // --- URLs Y CONSTANTES GLOBALES ---
    const API_EMPRESA = 'https://pale2.tufacturaya.com/restaurantepro/pedidos2/carta/Database/DataEmpresa.php';
    const API_PRODUCTOS = 'https://pale2.tufacturaya.com/restaurantepro/pedidos2/carta/Database/DataBase.php';
    const API_ENVIAR_PEDIDO = 'https://pale2.tufacturaya.com/restaurantepro/pedidos2/carta/Database/enviar-pedido.php'; 
    const LOGO_URL_BASE = 'https://pale2.tufacturaya.com/restaurantepro/facturacion/public/images/';
    const Img_URL = 'https://pale2.tufacturaya.com/restaurantepro/facturacion/public/images/productos/';
    
    let productosPolleria = [];
    let productosEnCarrito = JSON.parse(localStorage.getItem("productos-en-carrito")) || [];
    let enModoCheckout = false;

    // --- MAPEO DE ICONOS ---
    const iconosCategorias = { "POLLO": "bi-fire", "ALITAS": "bi-fan", "BEBIDAS": "bi-cup-straw", "GASEOSA": "bi-cup-straw", "JUGOS": "bi-cup-hot-fill", "CAFE": "bi-cup-hot-fill", "ADICIONALES": "bi-plus-square-dotted", "ENSALADA": "bi-egg-fried", "DEFAULT": "bi-tag-fill" };

    // --- SELECTORES DEL DOM ---
    const logoNavMenu = document.getElementById("logo-nav-menu"), 
          nombreNavMenu = document.getElementById("nombre-nav-menu"),
          listaProductos = document.querySelector("#lista-productos"), 
          menuCategorias = document.querySelector("#menu-categorias"), 
          inputBusqueda = document.querySelector("#input-busqueda"), 
          contCarritoProductos = document.querySelector("#carrito-productos"), 
          contCarritoVacio = document.querySelector("#carrito-vacio"), 
          totalElement = document.querySelector("#total"), 
          numeritoElement = document.querySelector("#numerito"), 
          btnComprar = document.querySelector("#carrito-acciones-comprar"), 
          btnVaciar = document.querySelector("#carrito-acciones-vaciar"), 
          cartPanel = document.querySelector(".cart-panel"), 
          cartFab = document.querySelector("#cart-fab"), 
          closeCartMobile = document.querySelector("#close-cart-mobile"), 
          mensajeCarga = document.querySelector("#mensaje-carga"),
          formCheckout = document.querySelector("#cart-checkout-form"),
          cartTitle = document.querySelector("#cart-title"),
          btnBackToCart = document.getElementById("btn-back-to-cart");
    
    // --- INICIALIZACIÓN ---
    async function init() {
        await Promise.all([
            cargarDatosEmpresa(),
            cargarDatosProductos()
        ]);
        renderizarCarrito();
        asignarEventosGenerales();
    }

    // --- LÓGICA DE CARGA DE DATOS ---
    async function cargarDatosEmpresa() {
        try {
            const response = await fetch(API_EMPRESA);
            if (!response.ok) throw new Error(`HTTP error! status: ${response.status}`);
            const data = await response.json();
            if (data && data.length > 0) {
                const empresa = data[0], nombreComercial = empresa.nombre_comercial || "Nombre no disponible";
                document.title = `Nuestra Carta - ${nombreComercial}`;
                if (empresa.logo) {
                    const logoUrl = LOGO_URL_BASE + empresa.logo;
                    if (logoNavMenu) { logoNavMenu.src = logoUrl; logoNavMenu.style.display = 'inline-block'; if (nombreNavMenu) nombreNavMenu.style.display = 'none'; }
                } else { if (nombreNavMenu) nombreNavMenu.innerText = nombreComercial; }
            }
        } catch (error) { 
            console.error("Error al cargar datos de la empresa en menu.html:", error); 
            if (nombreNavMenu) nombreNavMenu.innerText = "Mi Tienda";
        }
    }
    
    async function cargarDatosProductos() {
        try {
            const response = await fetch(API_PRODUCTOS);
            productosPolleria = await response.json();
            if (mensajeCarga) mensajeCarga.classList.add("disabled");
            renderizarCategorias();
            renderizarProductosAgrupados(productosPolleria);
        } catch (error) { 
            console.error("Error al cargar los productos:", error); 
            if(mensajeCarga) mensajeCarga.innerHTML = `<div class="alert alert-danger">Error al cargar la carta.</div>`;
        }
    }

    function renderizarCategorias() {
        const categoriasUnicas = ["Todos", ...new Set(productosPolleria.map(p => p.nombre_producto))];
        menuCategorias.innerHTML = categoriasUnicas.map(categoria => {
            const iconKey = Object.keys(iconosCategorias).find(key => categoria.toUpperCase().includes(key)) || "DEFAULT";
            const iconClass = iconosCategorias[iconKey];
            return `<div class="swiper-slide"><button class="categoria-btn ${categoria === 'Todos' ? 'active' : ''}" data-categoria="${categoria}"><i class="bi ${iconClass}"></i><span>${categoria}</span></button></div>`;
        }).join('');
        
        new Swiper('.categorias-swiper', {
            slidesPerView: 'auto',
            spaceBetween: 10,
            freeMode: true,
            navigation: {
                nextEl: '.categorias-button-next',
                prevEl: '.categorias-button-prev',
            },
        });
        
        asignarEventosCategorias();
    }

    function renderizarProductosAgrupados(productos) {
        listaProductos.innerHTML = "";
        const productosAgrupados = productos.reduce((acc, p) => { (acc[p.nombre_producto] = acc[p.nombre_producto] || []).push(p); return acc; }, {});
        for (const categoria in productosAgrupados) {
            const seccion = document.createElement("section");
            seccion.className = "categoria-seccion mb-5";
            seccion.dataset.categoria = categoria;
            seccion.innerHTML = `<h2 class="categoria-titulo-seccion">${categoria}</h2><div class="row row-cols-1 row-cols-md-2 row-cols-xl-3 g-4">${productosAgrupados[categoria].map(producto => `<div class="col producto-item" data-nombre="${producto.presentacion.toLowerCase()}"><div class="producto-card-vertical"><div class="producto-imagen-wrapper"><img class="producto-imagen-menu" src="${Img_URL + producto.archivo_imagen}" alt="${producto.presentacion}"></div><div class="producto-info-vertical"><h5 class="producto-titulo-menu">${producto.presentacion}</h5>${producto.descripcion ? `<p class="producto-descripcion">${producto.descripcion}</p>` : ''}<div class="producto-footer"><p class="producto-precio-menu mb-0">S/ ${parseFloat(producto.precio).toFixed(2)}</p><button class="btn-agregar-menu" data-id="${producto.id_pres}"><i class="bi bi-plus-lg"></i></button></div></div></div></div>`).join('')}</div>`;
            listaProductos.append(seccion);
        }
    }

    // --- LÓGICA DE VISUALIZACIÓN DE CHECKOUT ---
    function cambiarModoCheckout(activar) {
        enModoCheckout = activar;
        contCarritoProductos.classList.toggle('disabled', activar);
        formCheckout.classList.toggle('disabled', !activar);
        btnBackToCart.classList.toggle('disabled', !activar);
        if (contCarritoVacio) contCarritoVacio.classList.add('disabled');
        if (activar) {
            cartTitle.innerText = "Completa tus Datos";
            btnComprar.innerHTML = 'Confirmar y Enviar <i class="bi bi-check-lg"></i>';
            btnVaciar.innerText = "← Volver a Mi Pedido";
            btnVaciar.classList.remove('disabled');
        } else {
            cartTitle.innerText = "Mi Pedido";
            btnComprar.innerHTML = 'Finalizar Pedido <i class="bi bi-arrow-right"></i>';
            btnVaciar.innerText = "Vaciar Pedido";
            renderizarCarrito();
        }
    }
    
    // --- LÓGICA DEL CARRITO ---
    function renderizarCarrito() {
        const hayProductos = productosEnCarrito.length > 0;
        if(contCarritoVacio) contCarritoVacio.classList.toggle("disabled", hayProductos);
        if(btnComprar) btnComprar.classList.toggle("disabled", !hayProductos);
        if(btnVaciar) btnVaciar.classList.toggle("disabled", !hayProductos);
        if(cartFab) cartFab.classList.toggle("disabled", !hayProductos);
        if (contCarritoProductos) {
            contCarritoProductos.innerHTML = hayProductos ? productosEnCarrito.map(producto => {
                const subtotal = producto.precio * producto.cantidad;
                return `<div class="cart-item"><img class="cart-item-img" src="${Img_URL + producto.archivo_imagen}" alt="${producto.presentacion}"><div class="cart-item-info"><h6 class="cart-item-title">${producto.presentacion}</h6><p class="cart-item-price">S/ ${parseFloat(producto.precio).toFixed(2)}</p></div><div class="cart-item-actions"><button class="btn-qty decrease-qty" data-id="${producto.id_pres}"><i class="bi bi-dash-circle"></i></button><span>${producto.cantidad}</span><button class="btn-qty increase-qty" data-id="${producto.id_pres}"><i class="bi bi-plus-circle"></i></button></div><strong class="cart-item-subtotal">S/ ${subtotal.toFixed(2)}</strong></div>`;
            }).join('') : "";
        }
        actualizarTotalYNumerito();
    }
    
    function agregarAlCarrito(producto) {
        if (!producto) return;
        const index = productosEnCarrito.findIndex(p => p.id_pres == producto.id_pres);
        index > -1 ? productosEnCarrito[index].cantidad++ : productosEnCarrito.push({ ...producto, cantidad: 1 });
        Toastify({ text: `"${producto.presentacion}" agregado`, duration: 2000, gravity: "bottom", position: "left", style: { background: "var(--clr-dark)", borderRadius: "var(--border-radius-md)" } }).showToast();
        actualizarStorageYRenderizarCarrito();
    }
    
    function manejarDecrementarCantidad(id) {
        const index = productosEnCarrito.findIndex(p => p.id_pres == id);
        if (index !== -1) {
            productosEnCarrito[index].cantidad > 1 ? productosEnCarrito[index].cantidad-- : productosEnCarrito.splice(index, 1);
            actualizarStorageYRenderizarCarrito();
        }
    }
    
    function manejarIncrementarCantidad(id) {
        const index = productosEnCarrito.findIndex(p => p.id_pres == id);
        if (index !== -1) {
            productosEnCarrito[index].cantidad++;
            actualizarStorageYRenderizarCarrito();
        }
    }

    function actualizarStorageYRenderizarCarrito() {
        localStorage.setItem("productos-en-carrito", JSON.stringify(productosEnCarrito));
        renderizarCarrito();
    }

    function actualizarTotalYNumerito() {
        const total = productosEnCarrito.reduce((acc, p) => acc + (p.precio * p.cantidad), 0);
        const cantidadTotal = productosEnCarrito.reduce((acc, p) => acc + p.cantidad, 0);
        if (totalElement) totalElement.innerText = `S/ ${total.toFixed(2)}`;
        if (numeritoElement) numeritoElement.innerText = cantidadTotal;
    }

    function manejarBusqueda(e) {
        const texto = e.target.value.toLowerCase().trim();
        document.querySelectorAll(".producto-item").forEach(item => { item.style.display = item.dataset.nombre.includes(texto) ? 'block' : 'none'; });
        document.querySelectorAll(".categoria-seccion").forEach(seccion => { const productosVisibles = seccion.querySelectorAll(".producto-item[style*='block']"); seccion.style.display = productosVisibles.length > 0 ? 'block' : 'none'; });
    }

    // --- LÓGICA DE FINALIZACIÓN DE PEDIDO ---
    async function manejarFinalizarPedido() {
        if (!enModoCheckout) {
            cambiarModoCheckout(true);
        } else {
            const form = document.getElementById('checkout-form');
            if (!form.checkValidity()) {
                form.reportValidity();
                return;
            }

            btnComprar.disabled = true;
            btnComprar.innerHTML = 'Enviando... <span class="spinner-border spinner-border-sm"></span>';

            const datosCliente = {
                nombre: document.getElementById('cliente-nombre').value,
                telefono: document.getElementById('cliente-telefono').value,
                direccion: document.getElementById('cliente-direccion').value,
                referencia: document.getElementById('cliente-referencia').value,
                comentario: document.getElementById('cliente-comentario').value
            };

            const pedidoCompleto = {
                cliente: datosCliente,
                productos: productosEnCarrito,
                total: totalElement.innerText
            };

            try {
                const response = await fetch(API_ENVIAR_PEDIDO, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify(pedidoCompleto)
                });

                const result = await response.json();

                if (!response.ok || !result.success) {
                    throw new Error(result.message || 'Error desconocido en el servidor.');
                }

                Swal.fire('¡Pedido Enviado!', 'Gracias por tu compra, nos pondremos en contacto contigo pronto.', 'success');
                
                productosEnCarrito = [];
                form.reset();
                actualizarStorageYRenderizarCarrito();
                cambiarModoCheckout(false);

            } catch (error) {
                console.error("Error al enviar el pedido:", error);
                Swal.fire('¡Oops!', 'Hubo un problema al enviar tu pedido. Por favor, intenta de nuevo.', 'error');
            } finally {
                btnComprar.disabled = false;
            }
        }
    }

    // --- EVENTOS ---
    function asignarEventosGenerales() {
        inputBusqueda?.addEventListener("input", manejarBusqueda);
        cartFab?.addEventListener("click", () => cartPanel.classList.add("open"));
        closeCartMobile?.addEventListener("click", () => cartPanel.classList.remove("open"));
        btnComprar?.addEventListener("click", manejarFinalizarPedido);
        
        btnVaciar?.addEventListener("click", () => {
            if (enModoCheckout) {
                cambiarModoCheckout(false);
            } else {
                Swal.fire({title: '¿Vaciar Pedido?', text: "Se quitarán todos los productos.", icon: 'warning', showCancelButton: true, confirmButtonColor: '#d33', confirmButtonText: 'Sí, vaciar', cancelButtonText: 'Cancelar'}).then(result => { if (result.isConfirmed) { productosEnCarrito = []; actualizarStorageYRenderizarCarrito(); }});
            }
        });

        btnBackToCart?.addEventListener("click", () => {
            if (enModoCheckout) {
                cambiarModoCheckout(false);
            }
        });

        document.body.addEventListener("click", e => {
            const btnAgregar = e.target.closest(".btn-agregar-menu"); if (btnAgregar) { agregarAlCarrito(productosPolleria.find(p => p.id_pres == btnAgregar.dataset.id)); return; }
            const btnIncrementar = e.target.closest(".increase-qty"); if (btnIncrementar) { manejarIncrementarCantidad(btnIncrementar.dataset.id); return; }
            const btnDecrementar = e.target.closest(".decrease-qty"); if (btnDecrementar) { manejarDecrementarCantidad(btnDecrementar.dataset.id); }
        });
    }

    function asignarEventosCategorias() {
        menuCategorias.querySelectorAll(".categoria-btn").forEach(btn => {
            btn.addEventListener("click", e => {
                const currentBtn = e.currentTarget; if (currentBtn.classList.contains('active')) return;
                menuCategorias.querySelector(".active")?.classList.remove("active");
                currentBtn.classList.add("active");
                const categoria = currentBtn.dataset.categoria;
                document.querySelectorAll('.categoria-seccion').forEach(seccion => { seccion.style.display = (categoria === 'Todos' || seccion.dataset.categoria === categoria) ? 'block' : 'none'; });
            });
        });
    }
    
    init();
});